<?php

declare(strict_types=1);

namespace Erlage\Photogram\Tools;

use Erlage\Photogram\Settings;
use PHPMailer\PHPMailer\PHPMailer;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\User\UserModel;

/**
 * Quick Mailer for sending OTPs! 
 * 
 * * needs [refactoring] 
 * 
 * ? right now we've only two types of emails that server can dispatch, and both of them inclues
 * ? server pushing an OTP/or access token to user's email thus we decided to hardcode email template
 * ? rather than doing it in more optimal way i.e templating. this module will be refreshed in future
 * ? updates. few things that are on todo list with high-priority:
 * 
 * 1. decouple templating logic and add dynamic templates.
 * 
 * 2. move html markup and its contents to .html files. this is the only place in entire code base 
 *    where we've hardcoded html. 
 * 
 * 3. support for more types of emails(e.g liked your post, started following etc) and each email type
 *    should have its own template including its parsing rules
 * 
 * 3. admin's ability to change, alter specific email template
 * 
 * 4. [not on priority list but maybe] i10n support for emails.
*/
class OTPMailer
{
    /**
     * @var string
     */
    protected $subject = '';

    /**
     * @var string
     */
    protected $message = '';

    /**
     * @var string
     */
    protected $otp = '';

    /**
     * @var bool
     */
    protected $noReply = true;

    /**
     * @var UserModel
     */
    protected $userModel;

    public function setOTP(string $otp): self
    {
        // no need to sanitize here as otp can be set only via server code
        $this -> otp = '<br><br>
                            <h2 style="background: #a6a6a6;margin: 0 auto;width: max-content;letter-spacing:2px;padding: 10px;color: #fff;border-radius: 4px;">
                                ' . $otp . '
                            </h2>
                        <br><br>';

        return $this;
    }

    public function setUserModel(UserModel $userModel): self
    {
        $this -> userModel = $userModel;

        return $this;
    }

    public function dispatchRecoveryEmail()
    {
        $this -> subject = 'One time password for Recovering your Account';

        $this -> message = 'Please use the below code to complete your account recovery process.';

        $this -> dispatch();
    }

    public function dispatchVerificationEmail()
    {
        $this -> subject = 'Verify your email';

        $this -> message = 'Please use the below OTP code to complete your email verification process.';

        $this -> dispatch();
    }

    private function dispatch()
    {
        self::send(
            $this -> userModel,
            $this -> subject,
            $this -> parseTemplate(),
            $this -> noReply
        );
    }

    private static function send(
        UserModel $toUserModel,
        string $subject,
        string $content,
        bool $noReply = true
    ): void {
        /*
        |--------------------------------------------------------------------------
        | if smtp is enabled, user phpMailer
        |--------------------------------------------------------------------------
        */

        if (Settings::getBool(ServerConstants::SS_BOOL_MAILER_SMTP))
        {
            $phpMailer = new PHPMailer();

            $phpMailer -> IsSMTP();

            $phpMailer -> SMTPDebug = false;

            $phpMailer -> SMTPAuth = true;

            $phpMailer -> SMTPSecure = 'ssl';

            $phpMailer -> Host = Settings::getString(ServerConstants::SS_TEXT_MAILER_SMTP_HOST);

            $phpMailer -> Port = Settings::getString(ServerConstants::SS_INT_MAILER_SMTP_PORT);

            $phpMailer -> IsHTML(true);

            $phpMailer -> Username = Settings::getString(ServerConstants::SS_TEXT_MAILER_SMTP_USERNAME);

            $phpMailer -> Password = Settings::getString(ServerConstants::SS_TEXT_MAILER_SMTP_PASSWORD);

            $phpMailer -> SetFrom(
                ($noReply)
                    ? Settings::getString(ServerConstants::SS_TEXT_MAILER_NO_REPLY_EMAIL)
                    : Settings::getString(ServerConstants::SS_TEXT_MAILER_CONTACT_EMAIL),
                Settings::getString(ServerConstants::SS_TEXT_NAME)
            );

            $phpMailer -> Subject = $subject;

            $phpMailer -> Body = $content;

            $phpMailer -> AddAddress($toUserModel -> getEmail());

            $phpMailer -> Send();
        }

        /*
        |--------------------------------------------------------------------------
        | else use php's built in function mail()
        |--------------------------------------------------------------------------
        */

        else
        {
            $headers = 'MIME-Version: 1.0' . "\r\n";

            $headers .= 'Content-type: text/html; charset=utf-8' . "\r\n";

            $headers .= 'From: '
                    . Settings::getString(ServerConstants::SS_TEXT_NAME)

                    . ' <'

                    . (
                        ($noReply)
                            ? Settings::getString(ServerConstants::SS_TEXT_MAILER_NO_REPLY_EMAIL)
                            : Settings::getString(ServerConstants::SS_TEXT_MAILER_CONTACT_EMAIL)
                    )

                    . '>'

                    . PHP_EOL

                    . 'Reply-To: '

                    . Sanitizer::remove(Settings::getString(ServerConstants::SS_TEXT_NAME))

                    . ' <'

                    . (
                        ($noReply)
                            ? Settings::getString(ServerConstants::SS_TEXT_MAILER_NO_REPLY_EMAIL)
                            : Settings::getString(ServerConstants::SS_TEXT_MAILER_CONTACT_EMAIL)
                    )

                    . '>'
                    . PHP_EOL
                    . 'X-Mailer: PHP/' . PHP_VERSION;

            // dispatch email

            @\mail($toUserModel -> getEmail(), $subject, $content, $headers);
        }
    }

    private function parseTemplate(): string
    {
        $templateContent = self::hardcodedEmailTemplate();

        return \str_replace(
            array(
                '__OTP__',
                '__MESSAGE__',
                '__USERNAME__',
                '__SERVER_URL__',
                '__SERVER_NAME__',
                '__SS_EMAIL_FOOTER_ADDRESS__',
                '__SS_MAILER_CONTACT_EMAIL__',
                '__SS_EMAIL_HEADER_COLOR__',
            ),
            array(
                $this -> otp,
                $this -> message,
                $this -> userModel -> getUsername(),
                Settings::getString(ServerConstants::SS_TEXT_NAME),
                SERVER_URL,
                Settings::getString(ServerConstants::SS_TEXT_EMAIL_FOOTER),
                Settings::getString(ServerConstants::SS_TEXT_MAILER_CONTACT_EMAIL),
                Settings::getString(ServerConstants::SS_TEXT_EMAIL_HEADER),
            ),
            $templateContent
        );
    }

    /**
     * a complete templating solution will be added in future update and this mess will be removed.
     * 
     * for now we have only two types of emails and we don't want to spin up a full blown templating 
     * library for parsing just one single template. 
     */
    private static function hardcodedEmailTemplate(): string
    {
        $bottomPart = '<tr>
                            <td style="padding-left:20px;padding-right:20px;padding-bottom:0px;padding-top:20px">
                                <table width="100%" cellspacing="0" cellpadding="0" border="0" align="center">
                                    <tbody>
                                        <tr>
                                            <td style="text-align:left;padding-top:0px;padding-bottom:40px;padding-right:0;padding-left:0" valign="top">
                                                <p style="display:none"></p>
                                            </td>
                                        </tr>
                                    </tbody>
                                </table>
                            </td>
                        </tr>';

        $messagePart = '<td style="padding-left:10px;padding-right:10px">
                            <table style="font-size:16px;line-height:22px;color:#303030" width="100%" cellspacing="0" cellpadding="0" border="0">
                                <tbody>
                                    <tr>
                                        <td style="padding-top:20px;padding-bottom:0px;padding-right:10px;padding-left:10px" width="100%" valign="middle">
                                            <table style="font-size:16px;line-height:22px;color:#303030" width="100%" cellspacing="0" cellpadding="0" border="0">
                                                <tbody>
                                                    <tr>
                                                        <td style="padding-top:12px;padding-bottom:12px;padding-right:0px;padding-left:0" valign="middle">
                                                            <p style="margin-top:0;margin-bottom:30px!important;font-weight:500;font-family:-apple-system,BlinkMacSystemFont,\'Segoe UI\',Roboto,Oxygen-Sans,Ubuntu,Cantarell,\'Helvetica Neue\',sans-serif;font-size:38px;line-height:36px">
                                                                Hi __USERNAME__!
                                                            </p>
                                                            <span>
                                                                <p>
                                                                __MESSAGE__
                                                                </p>
                                                            </span> __OTP__
                                                        </td>
                                                    </tr>
                                                </tbody>
                                            </table>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </td>';

        $topPart = ' <div style="margin-top:0;margin-bottom:0;margin-right:auto;margin-left:auto;width:580px;min-width:580px">
                        <table style="margin-top:0;margin-bottom:0;margin-right:auto;margin-left:auto;width:580px;min-width:580px" width="100%" cellspacing="0" cellpadding="0" border="0" bgcolor="#FFFFFF">
                            <tbody>
                                <tr>
                                    <td>
                                        <table style="color:#303030" width="100%" cellspacing="0" cellpadding="0" border="0" align="center">
                                            <tbody>
                                                <tr>
                                                    ' . $messagePart . '
                                                </tr>
                                            </tbody>
                                        </table>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>';

        $bottom = '<table style="margin-top:0;margin-bottom:0;margin-right:auto;margin-left:auto;width:580px;min-width:580px" width="100%" cellspacing="0" cellpadding="0" border="0" bgcolor="#FFFFFF">
                    <tbody>
                        <tr>
                            <td>
                                <table width="100%" cellspacing="0" cellpadding="0" border="0" align="center">
                                    <tbody>
                                        <tr>
                                            <td style="padding-left:20px;padding-right:20px;padding-bottom:0px;padding-top:20px">
                                                <table style="border-top-width:2px;border-top-style:solid;border-top-color:#ececee;border-bottom-width:2px;border-bottom-style:solid;border-bottom-color:#ececee" width="100%" cellspacing="0" cellpadding="0" border="0" align="center">
                                                    <tbody>
                                                        <tr>
                                                            <td style="text-align:left;padding-top:25px;padding-bottom:40px;padding-right:0;padding-left:0" valign="top">
                                                                <p style="margin-top:0;margin-bottom:0;color:#959597;font-size:13px;padding-bottom:12px;font-weight:500"><a href="__SERVER_URL__" style="color:#959597;text-decoration:none" target="_blank" data-saferedirecturl="__SERVER_URL__">__SS_EMAIL_FOOTER_ADDRESS__<br>__SS_MAILER_CONTACT_EMAIL__</a></p>
                                                            </td>
                                                        </tr>
                                                    </tbody>
                                                </table>
                                            </td>
                                        </tr>
                                        ' . $bottomPart . '
                                    </tbody>
                                </table>
                            </td>
                        </tr>
                    </tbody>
                </table>';

        return '
        <div style="margin-top:0;margin-bottom:0;margin-right:0;
            margin-left:0;padding-top:0;padding-bottom:0;
            padding-right:0;padding-left:0;background-color:__SS_EMAIL_HEADER_COLOR__;
            ">
                <table style="width:580px;min-width:580px" cellspacing="0" cellpadding="0" border="0">
                    <tbody>
                        <tr>
                            <td style="font-size:1px;line-height:1px;min-width:580px" width="580"></td>
                        </tr>
                    </tbody>
                </table>
                <div> &nbsp;
                    <table style="margin-top:0;margin-bottom:0;margin-right:auto;margin-left:auto;table-layout:fixed;width:100%;border-collapse:separate;text-align:left;color:#303030;font-family:-apple-system,BlinkMacSystemFont,\'Segoe UI\',Roboto,sans-serif" width="100%"
                        cellspacing="0" cellpadding="0" border="0" bgcolor="#FFFFFF" align="center">
                        <tbody>
                            <tr>
                                <td align="center">
                                    <center>
                                       ' . $topPart . '
                                    </center>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                    <table style="margin-top:0;margin-bottom:0;margin-right:auto;margin-left:auto;table-layout:fixed;width:100%;border-collapse:separate;text-align:left;color:#303030;font-family:-apple-system,BlinkMacSystemFont,\'Segoe UI\',Roboto,sans-serif" width="100%"
                        cellspacing="0" cellpadding="0" border="0" bgcolor="#FFFFFF" align="center">
                        <tbody>
                            <tr>
                                <td align="center">
                                    <center>
                                        <div style="margin-top:0;margin-bottom:0;margin-right:auto;margin-left:auto;width:580px;min-width:580px"> </div>
                                    </center>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                    <table style="margin-top:0;margin-bottom:0;margin-right:auto;margin-left:auto;table-layout:fixed;width:100%;border-collapse:separate;text-align:left;color:#303030;font-family:-apple-system,BlinkMacSystemFont,\'Segoe UI\',Roboto,sans-serif" width="100%"
                        cellspacing="0" cellpadding="0" border="0" bgcolor="#FFFFFF" align="center">
                        <tbody>
                            <tr>
                                <td align="center">
                                    <center>
                                        <div style="margin-top:0;margin-bottom:0;margin-right:auto;margin-left:auto;width:580px;min-width:580px">
                                            ' . $bottom . '
                                        </div>
                                    </center>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

        ';
    }
}
